<?php

use LAM\PDF\PDFTable;
use LAM\PDF\PDFTableCell;
use LAM\PDF\PDFTableRow;
use LAM\TYPES\ConfiguredType;
use LAM\TYPES\TypeManager;

/*

  This code is part of LDAP Account Manager (http://www.ldap-account-manager.org/)
  Copyright (C) 2013 - 2025  Roland Gruber

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/**
 * Manages Windows AD (e.g. Samba 4) groups.
 *
 * @package modules
 * @author Roland Gruber
 */

/**
 * Manages Windows AD (e.g. Samba 4) groups.
 *
 * @package modules
 */
class windowsGroup extends baseModule {

	/**
	 * These attributes will be ignored by default if a new account is copied from an existing one.
	 */
	private const ATTRIBUTES_TO_IGNORE_ON_COPY = ['sAMAccountName', 'msSFU30Name', 'msSFU30NisDomain'];

	/** possible group types (e.g. distribution) */
	private $groupTypes;
	/** possible group scopes (e.g. universal) */
	private $groupScopes;
	/** group cache */
	private $groupCache;

	/** security group */
	const TYPE_SECURITY = 'security';
	/** email list */
	const TYPE_DISTRIBUTION = 'distribution';
	/** domain local group */
	const SCOPE_DOMAIN_LOCAL = 'domain';
	/** global group */
	const SCOPE_GLOBAL = 'global';
	/** universal group */
	const SCOPE_UNIVERSAL = 'universal';

	/**
	 * Creates a new module for Samba 3 groups.
	 *
	 * @param string $scope account type
	 */
	function __construct($scope) {
		$this->groupTypes = [
			_('Security') => windowsGroup::TYPE_SECURITY,
			_('Distribution') => windowsGroup::TYPE_DISTRIBUTION,
		];
		$this->groupScopes = [
			_('Domain local') => windowsGroup::SCOPE_DOMAIN_LOCAL,
			_('Global') => windowsGroup::SCOPE_GLOBAL,
			_('Universal') => windowsGroup::SCOPE_UNIVERSAL,
		];
		// call parent constructor
		parent::__construct($scope);
	}

	/**
	 * Returns true if this module can manage accounts of the current type, otherwise false.
	 *
	 * @return boolean true if module fits
	 */
	public function can_manage() {
		return $this->get_scope() === 'group';
	}

	/**
	 * Returns meta data that is interpreted by parent class
	 *
	 * @return array array with meta data
	 *
	 * @see baseModule::get_metaData()
	 */
	public function get_metaData() {
		$return = [];
		// icon
		$return['icon'] = 'samba.svg';
		// this is a base module
		$return["is_base"] = true;
		// RDN attribute
		$return["RDN"] = ["cn" => "high"];
		// LDAP filter
		$return["ldap_filter"] = ['and' => "", 'or' => '(objectClass=group)'];
		// alias name
		$return["alias"] = _("Windows");
		// module dependencies
		$return['dependencies'] = ['depends' => [], 'conflicts' => []];
		// managed object classes
		$return['objectClasses'] = ['group', 'securityPrincipal', 'mailRecipient'];
		// managed attributes
		$return['attributes'] = ['cn', 'description', 'info', 'mail', 'member', 'memberOf', 'sAMAccountName',
			'groupType', 'managedBy', 'msSFU30Name', 'msSFU30NisDomain'];
		// help Entries
		$return['help'] = [
			'hiddenOptions' => [
				"Headline" => _("Hidden options"),
				"Text" => _("The selected options will not be managed inside LAM. You can use this to reduce the number of displayed input fields.")
			],
			'cn' => [
				"Headline" => _('Group name'), 'attr' => 'cn, sAMAccountName',
				"Text" => _('Please enter the group name.')
			],
			'description' => [
				"Headline" => _('Description'), 'attr' => 'description',
				"Text" => _('Please enter a descriptive text for this group.')
			],
			'info' => [
				"Headline" => _('Notes'), 'attr' => 'info',
				"Text" => _('Additional notes to describe this entry.')
			],
			'mail' => [
				"Headline" => _('Email address'), 'attr' => 'mail',
				"Text" => _('The list\'s email address.')
			],
			'member' => [
				"Headline" => _('Members'), 'attr' => 'member',
				"Text" => _('This is a list of members of this group.')
			],
			'memberOf' => [
				"Headline" => _('Member of'), 'attr' => 'memberOf',
				"Text" => _('This is a list of groups this group is member of.')
			],
			'memberList' => [
				"Headline" => _('Members'), 'attr' => 'member',
				"Text" => _('This is a list of members of this group. Multiple members are separated by semicolons.')
			],
			'groupType' => [
				"Headline" => _('Group type'), 'attr' => 'groupType',
				"Text" => _('Security groups are used for permission management and distribution groups as email lists.')
			],
			'groupScope' => [
				"Headline" => _('Group scope'), 'attr' => 'groupType',
				"Text" => _('Please specify the group scope.')
			],
			'managedBy' => [
				"Headline" => _('Managed by'), 'attr' => 'managedBy',
				"Text" => _('The group is managed by this contact person.')
			],
			'msSFU30Name' => [
				"Headline" => _('NIS name'), 'attr' => 'msSFU30Name',
				"Text" => _('Group name for NIS.')
			],
			'msSFU30NisDomain' => [
				"Headline" => _('NIS domain'), 'attr' => 'msSFU30NisDomain',
				"Text" => _('NIS domain name.')
			],
			'filter' => [
				"Headline" => _("Filter"),
				"Text" => _("Here you can enter a filter value. Only entries which contain the filter text will be shown.")
					. ' ' . _('Possible wildcards are: "*" = any character, "^" = line start, "$" = line end')
			],
		];
		// upload fields
		$return['upload_columns'] = [
			[
				'name' => 'windowsGroup_name',
				'description' => _('Group name'),
				'help' => 'cn',
				'example' => _('Domain administrators'),
				'required' => true
			],
			[
				'name' => 'windowsGroup_description',
				'description' => _('Description'),
				'help' => 'description',
				'example' => _('Domain administrators'),
			],
			[
				'name' => 'windowsGroup_notes',
				'description' => _('Notes'),
				'help' => 'info',
				'example' => _('Domain administrators'),
			],
			[
				'name' => 'windowsGroup_scope',
				'description' => _('Group scope'),
				'help' => 'groupScope',
				'values' => implode(', ', array_values($this->groupScopes)),
				'example' => windowsGroup::SCOPE_GLOBAL,
				'default' => windowsGroup::SCOPE_GLOBAL,
			],
			[
				'name' => 'windowsGroup_type',
				'description' => _('Group type'),
				'help' => 'groupType',
				'values' => implode(', ', array_values($this->groupTypes)),
				'example' => windowsGroup::TYPE_SECURITY,
				'default' => windowsGroup::TYPE_SECURITY,
			],
			[
				'name' => 'windowsGroup_members',
				'description' => _('Members'),
				'help' => 'memberList',
				'example' => 'uid=user1,o=test;uid=user2,o=test',
			],
		];
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemail')) {
			$return['upload_columns'][] = [
				'name' => 'windowsGroup_mail',
				'description' => _('Email address'),
				'help' => 'mail',
				'example' => _('group@company.com'),
			];
		}
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemanagedBy')) {
			$return['upload_columns'][] = [
				'name' => 'windowsGroup_managedBy',
				'description' => _('Managed by'),
				'help' => 'managedBy',
				'example' => 'cn=user1,o=test',
			];
		}
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30Name', true)) {
			$return['upload_columns'][] = [
				'name' => 'windowsGroup_msSFU30Name',
				'description' => _('NIS name'),
				'help' => 'msSFU30Name',
				'example' => _('adminstrators'),
			];
		}
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30NisDomain', true)) {
			$return['upload_columns'][] = [
				'name' => 'windowsGroup_msSFU30NisDomain',
				'description' => _('NIS domain'),
				'help' => 'msSFU30NisDomain',
				'example' => _('domain'),
			];
		}
		// profile options
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30NisDomain', true)) {
			$profileContainer = new htmlResponsiveRow();
			$profileContainer->add(new htmlResponsiveInputField(_('NIS domain'), 'windowsGroup_msSFU30NisDomain', null, 'msSFU30NisDomain'));
			$return['profile_options'] = $profileContainer;
			$return['profile_mappings']['windowsGroup_msSFU30NisDomain'] = 'msSFU30NisDomain';
		}
		// available PDF fields
		$return['PDF_fields'] = [
			'cn' => _('Group name'),
			'description' => _('Description'),
			'info' => _('Notes'),
			'member' => _('Members'),
			'memberOf' => _('Member of'),
			'groupType' => _('Group type'),
			'groupScope' => _('Group scope'),
		];
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemail')) {
			$return['PDF_fields']['mail'] = _('Email address');
		}
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemanagedBy')) {
			$return['PDF_fields']['managedBy'] = _('Managed by');
		}
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30Name', true)) {
			$return['PDF_fields']['msSFU30Name'] = _('NIS name');
		}
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30NisDomain', true)) {
			$return['PDF_fields']['msSFU30NisDomain'] = _('NIS domain');
		}
		return $return;
	}

	/**
	 * This function fills the $messages variable with output messages from this module.
	 */
	public function load_Messages() {
		$this->messages['cn'][0] = ['ERROR', _('Group name'), _('Group name contains invalid characters. Valid characters are: a-z, A-Z, 0-9 and .-_ !')];
		$this->messages['cn'][1] = ['ERROR', _('Account %s:') . ' windowsGroup_cn', _('Group name contains invalid characters. Valid characters are: a-z, A-Z, 0-9 and .-_ !')];
		$this->messages['mail'][0] = ['ERROR', _('Email address'), _('Please enter a valid email address!')];
		$this->messages['mail'][1] = ['ERROR', _('Account %s:') . ' windowsGroup_mail', _('Please enter a valid email address!')];
		$this->messages['groupScope'][0] = ['ERROR', _('Account %s:') . ' windowsGroup_groupScope', _('Please enter a valid group scope.')];
		$this->messages['groupType'][0] = ['ERROR', _('Account %s:') . ' windowsGroup_groupType', _('Please enter a valid group type.')];
		$this->messages['msSFU30Name'][0] = ['ERROR', _('NIS name'), _('NIS name contains invalid characters. Valid characters are: a-z, A-Z, 0-9 and .-_ !')];
		$this->messages['msSFU30Name'][1] = ['ERROR', _('Account %s:') . ' windowsGroup_msSFU30Name', _('NIS name contains invalid characters. Valid characters are: a-z, A-Z, 0-9 and .-_ !')];
	}

	/**
	 * {@inheritDoc}
	 */
	public function loadAttributesFromAccountCopy(array $ldapAttributes, array $attributesToIgnore = []): void {
		$attributesToIgnore = array_merge(baseModule::ATTRIBUTES_TO_IGNORE_ON_COPY_DEFAULT, self::ATTRIBUTES_TO_IGNORE_ON_COPY);
		parent::loadAttributesFromAccountCopy($ldapAttributes, $attributesToIgnore);
	}

	/**
	 * Returns the HTML meta data for the main account page.
	 *
	 * @return htmlElement HTML meta data
	 */
	public function display_html_attributes() {
		$container = new htmlResponsiveRow();
		$this->addSimpleInputTextField($container, 'cn', _('Group name'), true);
		$this->addSimpleInputTextField($container, 'description', _('Description'));
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemail')) {
			$this->addSimpleInputTextField($container, 'mail', _('Email address'));
		}
		// group type
		$groupType = windowsGroup::TYPE_SECURITY;
		$groupScope = windowsGroup::SCOPE_GLOBAL;
		if (isset($this->attributes['groupType'][0])) {
			if ((((int) $this->attributes['groupType'][0]) & 4) !== 0) {
				$groupScope = windowsGroup::SCOPE_DOMAIN_LOCAL;
			}
			elseif ((((int) $this->attributes['groupType'][0]) & 8) !== 0) {
				$groupScope = windowsGroup::SCOPE_UNIVERSAL;
			}
			if ((((int) $this->attributes['groupType'][0]) & 0x80000000) === 0) {
				$groupType = windowsGroup::TYPE_DISTRIBUTION;
			}
		}
		$scopeList = $this->groupScopes;
		// do not allow invalid conversions
		if (isset($this->orig['groupType'][0])) {
			$flippedScopes = array_flip($this->groupScopes);
			if ((((int) $this->orig['groupType'][0]) & 2) !== 0) {
				// no change from global to domain local
				unset($scopeList[$flippedScopes[windowsGroup::SCOPE_DOMAIN_LOCAL]]);
			}
			elseif ((((int) $this->orig['groupType'][0]) & 4) !== 0) {
				// no change from domain local to global
				unset($scopeList[$flippedScopes[windowsGroup::SCOPE_GLOBAL]]);
			}
		}
		$groupScopeSelect = new htmlResponsiveSelect('groupScope', $scopeList, [$groupScope], _('Group scope'), 'groupScope');
		$groupScopeSelect->setHasDescriptiveElements(true);
		$container->add($groupScopeSelect);
		$groupTypeSelect = new htmlResponsiveSelect('groupType', $this->groupTypes, [$groupType], _('Group type'), 'groupType');
		$groupTypeSelect->setHasDescriptiveElements(true);
		$container->add($groupTypeSelect);
		// notes
		$this->addSimpleInputTextField($container, 'info', _('Notes'), false, null, true);
		// managed by
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemanagedBy')) {
			$container->addLabel(new htmlOutputText(_('Managed by')));
			$managedBy = '-';
			if (isset($this->attributes['managedBy'][0])) {
				$managedBy = $this->attributes['managedBy'][0];
			}
			$managedByGroup = new htmlGroup();
			$managedByGroup->addElement(new htmlOutputText(getAbstractDN($managedBy)));
			$managedByGroup->addElement(new htmlSpacer('0.5rem', null));
			$managedByGroup->addElement(new htmlHelpLink('managedBy'));
			$container->addField($managedByGroup);
			$managedByButtons = new htmlGroup();
			$managedByButtons->addElement(new htmlAccountPageButton(static::class, 'managedBy', 'edit', _('Change')));
			if (isset($this->attributes['managedBy'][0])) {
				$managedByButtons->addElement(new htmlSpacer('5px', null));
				$managedByButtons->addElement(new htmlAccountPageButton(static::class, 'attributes', 'removeManagedBy', _('Remove')));
			}
			$container->addLabel(new htmlOutputText('&nbsp;', false));
			$container->addField($managedByButtons);
		}
		// NIS
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30Name', true) || !$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30NisDomain', true)) {
			$container->add(new htmlSubTitle(_('NIS')));
			if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30Name', true)) {
				$this->addSimpleInputTextField($container, 'msSFU30Name', _('NIS name'));
			}
			if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30NisDomain', true)) {
				$this->addSimpleInputTextField($container, 'msSFU30NisDomain', _('NIS domain'));
			}
			$container->addVerticalSpacer('2rem');
		}
		// group members
		$container->addVerticalSpacer('1rem');
		$container->addLabel(new htmlOutputText(_("Group members")));
		$memberButtons = new htmlGroup();
		$memberButtons->addElement(new htmlAccountPageButton(static::class, 'user', 'open', _('Edit')));
		if (!empty($this->attributes['member'])) {
			$memberButtons->addElement(new htmlSpacer('0.5rem', null));
			$memberButtons->addElement(new htmlAccountPageButton(static::class, 'effectiveMembers', 'open', _('Show effective members')));
		}
		$memberButtons->addElement(new htmlSpacer('0.5rem', null));
		$memberButtons->addElement(new htmlHelpLink('member'));
		$container->addField($memberButtons);
		$memberList = [];
		if (isset($this->attributes['member'])) {
			for ($i = 0; $i < count($this->attributes['member']); $i++) {
				$memberList[] = $this->attributes['member'][$i];
			}
			usort($memberList, compareDN(...));
		}
		$members = new htmlTable();
		$members->alignment = htmlElement::ALIGN_RIGHT;
		$members->setCSSClasses(['fullwidth']);
		for ($i = 0; $i < count($memberList); $i++) {
			$member = new htmlOutputText(getAbstractDN($memberList[$i]));
			$member->alignment = htmlElement::ALIGN_RIGHT;
			$members->addElement($member, true);
		}
		$container->addLabel(new htmlOutputText('&nbsp;', false));
		$container->addField($members);
		// member of
		$container->addVerticalSpacer('2rem');
		$container->addLabel(new htmlOutputText(_("Member of")));
		$memberOfGroup = new htmlGroup();
		$memberOfGroup->addElement(new htmlAccountPageButton(static::class, 'memberof', 'open', _('Edit')));
		$memberOfGroup->addElement(new htmlSpacer('0.5rem', null));
		$memberOfGroup->addElement(new htmlHelpLink('memberOf'));
		$container->addField($memberOfGroup);
		$memberList = [];
		if (isset($this->attributes['memberOf'])) {
			for ($i = 0; $i < count($this->attributes['memberOf']); $i++) {
				$memberList[] = $this->attributes['memberOf'][$i];
			}
			usort($memberList, compareDN(...));
		}
		$memberOf = new htmlTable();
		$memberOf->setCSSClasses(['fullwidth']);
		$memberOf->alignment = htmlElement::ALIGN_RIGHT;
		for ($i = 0; $i < count($memberList); $i++) {
			$member = new htmlOutputText(getAbstractDN($memberList[$i]));
			$member->alignment = htmlElement::ALIGN_RIGHT;
			$memberOf->addElement($member, true);
		}
		$container->addLabel(new htmlOutputText('&nbsp;', false));
		$container->addField($memberOf);

		return $container;
	}

	/**
	 * Processes user input of the primary module page.
	 * It checks if all input values are correct and updates the associated LDAP attributes.
	 *
	 * @return array list of info/error messages
	 */
	public function process_attributes() {
		$return = [];
		// cn
		$this->attributes['cn'][0] = $_POST['cn'];
		$this->attributes['sAMAccountName'][0] = $_POST['cn'];
		if (!get_preg($_POST['cn'], 'groupname')) {
			$return[] = $this->messages['cn'][0];
		}
		// description
		$this->attributes['description'][0] = $_POST['description'];
		// email
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemail')) {
			$this->attributes['mail'][0] = $_POST['mail'];
			if (!empty($_POST['mail']) && !get_preg($_POST['mail'], 'email')) {
				$return[] = $this->messages['mail'][0];
			}
		}
		// group scope
		switch ($_POST['groupScope']) {
			case windowsGroup::SCOPE_DOMAIN_LOCAL:
				$this->attributes['groupType'][0] = 4;
				break;
			case windowsGroup::SCOPE_GLOBAL:
				$this->attributes['groupType'][0] = 2;
				break;
			case windowsGroup::SCOPE_UNIVERSAL:
				$this->attributes['groupType'][0] = 8;
				break;
		}
		// group type
		if ($_POST['groupType'] == windowsGroup::TYPE_SECURITY) {
			$this->attributes['groupType'][0] -= 2147483648;
		}
		// notes
		$this->attributes['info'][0] = $_POST['info'];
		// managed by
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemanagedBy') && isset($_POST['form_subpage_' . static::class . '_attributes_removeManagedBy'])) {
			unset($this->attributes['managedBy']);
		}
		// NIS name
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30Name', true)) {
			if ($this->getAccountContainer()->isNewAccount && !isset($this->attributes['msSFU30Name']) && empty($_POST['msSFU30Name'])) {
				$this->attributes['msSFU30Name'][0] = $_POST['cn'];
			}
			else {
				$this->attributes['msSFU30Name'][0] = $_POST['msSFU30Name'];
			}
			if (!empty($this->attributes['msSFU30Name'][0]) && !get_preg($this->attributes['msSFU30Name'][0], 'groupname')) {
				$return[] = $this->messages['msSFU30Name'][0];
			}
		}
		// NIS domain
		if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30Name', true)) {
			$this->attributes['msSFU30NisDomain'][0] = $_POST['msSFU30NisDomain'];
		}
		return $return;
	}

	/**
	 * Displays the memberof selection.
	 *
	 * @return htmlElement meta HTML code
	 */
	function display_html_memberof() {
		$return = new htmlResponsiveRow();
		$return->add(new htmlSubTitle(_("Groups")));
		$groups = $this->findGroups();
		// sort by DN
		usort($groups, compareDN(...));

		$selectedGroups = [];
		if (empty($this->attributes['memberOf'])) {
			$this->attributes['memberOf'] = [];
		}
		// sort by DN
		usort($this->attributes['memberOf'], compareDN(...));
		for ($i = 0; $i < count($this->attributes['memberOf']); $i++) {
			if (in_array($this->attributes['memberOf'][$i], $groups)) {
				$selectedGroups[getAbstractDN($this->attributes['memberOf'][$i])] = $this->attributes['memberOf'][$i];
			}
		}
		$availableGroups = [];
		foreach ($groups as $dn) {
			if (!in_array($dn, $this->attributes['memberOf'])) {
				$availableGroups[getAbstractDN($dn)] = $dn;
			}
		}

		$this->addDoubleSelectionArea($return, _("Selected groups"), _("Available groups"), $selectedGroups,
			[], $availableGroups, [], 'memberof', false, true);

		$return->addVerticalSpacer('2rem');
		$backButton = new htmlAccountPageButton(static::class, 'attributes', 'back', _('Back'));
		$return->add($backButton);

		return $return;
	}

	/**
	 * Processes user input of the memberof selection page.
	 * It checks if all input values are correct and updates the associated LDAP attributes.
	 *
	 * @return array list of info/error messages
	 */
	function process_memberof() {
		if (isset($_POST['memberof_2']) && isset($_POST['memberof_left'])) { // Add groups to list
			// add new group
			$this->attributes['memberOf'] = @array_merge($this->attributes['memberOf'], $_POST['memberof_2']);
		}
		elseif (isset($_POST['memberof_1']) && isset($_POST['memberof_right'])) { // remove groups from list
			$this->attributes['memberOf'] = array_delete($_POST['memberof_1'], $this->attributes['memberOf']);
		}
		return [];
	}

	/**
	 * This function will create the meta HTML code to show a page to change the member attribute.
	 *
	 * @return htmlElement HTML meta data
	 */
	function display_html_managedBy() {
		$return = new htmlResponsiveRow();
		// show possible managers
		$options = [];
		$filter = get_ldap_filter('user');
		$entries = searchLDAPByFilter($filter, ['dn'], ['user']);
		for ($i = 0; $i < count($entries); $i++) {
			$entries[$i] = $entries[$i]['dn'];
		}
		// sort by DN
		usort($entries, compareDN(...));
		for ($i = 0; $i < count($entries); $i++) {
			$options[getAbstractDN($entries[$i])] = $entries[$i];
		}
		$selected = [];
		if (isset($this->attributes['managedBy'][0])) {
			$selected = [$this->attributes['managedBy'][0]];
			if (!in_array($selected[0], $options)) {
				$options[getAbstractDN($selected[0])] = $selected[0];
			}
		}
		$membersSelect = new htmlSelect('managedBy', $options, $selected);
		$membersSelect->setHasDescriptiveElements(true);
		$membersSelect->setRightToLeftTextDirection(true);
		$membersSelect->setSortElements(false);
		$membersSelect->setTransformSingleSelect(false);
		$return->add($membersSelect);
		$filterGroup = new htmlGroup();
		$filterGroup->addElement(new htmlOutputText(_('Filter')));
		$filterInput = new htmlInputField('filter');
		$filterInput->filterSelectBox('managedBy');
		$filterInput->setCSSClasses(['max-width-10']);
		$filterGroup->addElement($filterInput);
		$filterGroup->addElement(new htmlHelpLink('filter'));
		$return->add($filterGroup);
		$return->addVerticalSpacer('2rem');
		$buttonTable = new htmlGroup();
		$buttonTable->addElement(new htmlAccountPageButton(static::class, 'attributes', 'set', _('Change')));
		$buttonTable->addElement(new htmlSpacer('0.5rem', null));
		$buttonTable->addElement(new htmlAccountPageButton(static::class, 'attributes', 'cancel', _('Cancel')));
		$return->add($buttonTable);
		return $return;
	}

	/**
	 * Processes user input of the members page.
	 * It checks if all input values are correct and updates the associated LDAP attributes.
	 *
	 * @return array list of info/error messages
	 */
	function process_managedBy() {
		$return = [];
		if (isset($_POST['form_subpage_' . static::class . '_attributes_set'])) {
			$this->attributes['managedBy'][0] = $_POST['managedBy'];
		}
		return $return;
	}

	/**
	 * This function will create the meta HTML code to show a page to change the member attribute.
	 *
	 * @return htmlElement HTML meta data
	 */
	function display_html_user() {
		$return = new htmlResponsiveRow();
		$typeManager = new TypeManager();
		// show list of possible new members
		if (isset($_POST['form_subpage_' . static::class . '_user_select']) && isset($_POST['type'])) {
			$type = $typeManager->getConfiguredType($_POST['type']);
			$filterGroup = new htmlGroup();
			$filterGroup->addElement(new htmlOutputText(_('Filter') . ' '));
			$filter = new htmlInputField('windows_filter');
			$filter->filterSelectBox('members');
			$filterGroup->addElement($filter);
			$return->add($filterGroup);
			$return->addVerticalSpacer('1rem');

			$options = [];
			$filter = get_ldap_filter($type->getId());
			$entries = searchLDAP($type->getSuffix(), $filter, ['dn']);
			for ($i = 0; $i < count($entries); $i++) {
				$entries[$i] = $entries[$i]['dn'];
			}
			// sort by DN
			usort($entries, compareDN(...));
			for ($i = 0; $i < count($entries); $i++) {
				if (!isset($this->attributes['member']) || !in_array($entries[$i], $this->attributes['member'])) {
					$options[getAbstractDN($entries[$i])] = $entries[$i];
				}
			}
			$size = 20;
			if (count($options) < 20) {
				$size = count($options);
			}
			$membersSelect = new htmlSelect('members', $options, [], $size);
			$membersSelect->setHasDescriptiveElements(true);
			$membersSelect->setMultiSelect(true);
			$membersSelect->setRightToLeftTextDirection(true);
			$membersSelect->setSortElements(false);
			$membersSelect->setTransformSingleSelect(false);
			$return->add($membersSelect);
			$return->addVerticalSpacer('2rem');
			$buttonTable = new htmlGroup();
			$buttonTable->addElement(new htmlAccountPageButton(static::class, 'user', 'addMembers', _('Add')));
			$buttonTable->addElement(new htmlAccountPageButton(static::class, 'user', 'cancel', _('Cancel')));
			$return->add($buttonTable);
			return $return;
		}
		// show existing members
		$membersTemp = [];
		if (isset($this->attributes['member'])) {
			$membersTemp = $this->attributes['member'];
		}
		// sort by DN
		usort($membersTemp, compareDN(...));
		$members = [];
		for ($i = 0; $i < count($membersTemp); $i++) {
			$members[getAbstractDN($membersTemp[$i])] = $membersTemp[$i];
		}
		$size = 20;
		if (isset($this->attributes['member']) && (count($this->attributes['member']) < 20)) {
			$size = count($this->attributes['member']);
		}
		if ($members !== []) {
			$membersSelect = new htmlSelect('members', $members, [], $size);
			$membersSelect->setHasDescriptiveElements(true);
			$membersSelect->setMultiSelect(true);
			$membersSelect->setRightToLeftTextDirection(true);
			$membersSelect->setSortElements(false);
			$membersSelect->setTransformSingleSelect(false);
			$return->add($membersSelect);
			$return->addVerticalSpacer('0.5rem');
			$removeButton = new htmlAccountPageButton(static::class, 'user', 'remove', _('Remove selected entries'));
			$return->add($removeButton);
			$return->addVerticalSpacer('2rem');
			$return->add(new htmlHorizontalLine());
			$return->addVerticalSpacer('2rem');
		}
		$types = $typeManager->getConfiguredTypes();
		$options = [];
		$optionsSelected = [];
		foreach ($types as $type) {
			$options[$type->getAlias()] = $type->getId();
			if ($type->getScope() == 'user') {
				$optionsSelected[] = $type->getId();
			}
		}
		$return->addLabel(new htmlOutputText(_('Add entries of this type:')));
		$typeSelect = new htmlSelect('type', $options, $optionsSelected);
		$typeSelect->setHasDescriptiveElements(true);
		$return->addField($typeSelect);
		$return->addLabel(new htmlOutputText('&nbsp;', false));
		$return->addField(new htmlAccountPageButton(static::class, 'user', 'select', _('Ok')));
		$return->addVerticalSpacer('2rem');
		$return->add(new htmlAccountPageButton(static::class, 'attributes', 'membersBack', _('Back')));
		return $return;
	}

	/**
	 * Processes user input of the members page.
	 * It checks if all input values are correct and updates the associated LDAP attributes.
	 *
	 * @return array list of info/error messages
	 */
	function process_user() {
		$return = [];
		if (isset($_POST['form_subpage_' . static::class . '_user_remove']) && isset($_POST['members'])) {
			$members = array_flip($this->attributes['member']);
			for ($i = 0; $i < count($_POST['members']); $i++) {
				if (isset($members[$_POST['members'][$i]])) {
					unset($members[$_POST['members'][$i]]);
				}
			}
			$this->attributes['member'] = array_values(array_flip($members));
		}
		elseif (isset($_POST['form_subpage_' . static::class . '_user_addMembers']) && isset($_POST['members'])) {
			for ($i = 0; $i < count($_POST['members']); $i++) {
				$this->attributes['member'][] = $_POST['members'][$i];
				$this->attributes['member'] = array_unique($this->attributes['member']);
			}
		}
		return $return;
	}

	/**
	 * This function will create the meta HTML code to show a page to list effective members.
	 *
	 * @return htmlElement HTML meta data
	 */
	function display_html_effectiveMembers() {
		$return = new htmlResponsiveRow();
		$effectiveMembers = $this->getEffectiveMembers();
		$entryTable = new htmlTable();
		// sort by DN, align right
		usort($effectiveMembers, compareDN(...));
		$entryTable->alignment = htmlElement::ALIGN_RIGHT;
		foreach ($effectiveMembers as $member) {
			$entry = new htmlOutputText(getAbstractDN($member));
			$entry->alignment = htmlElement::ALIGN_RIGHT;
			$entryTable->addElement($entry, true);
		}
		$return->add($entryTable);
		$return->addVerticalSpacer('2rem');
		$return->add(new htmlAccountPageButton(static::class, 'attributes', 'membersEffectiveBack', _('Back')));
		return $return;
	}

	/**
	 * Processes user input of the effective members page.
	 *
	 * @return array list of info/error messages
	 */
	function process_effectiveMembers() {
		// no processing, page is read-only
		return [];
	}

	/**
	 * {@inheritDoc}
	 * @see baseModule::build_uploadAccounts()
	 */
	public function build_uploadAccounts($rawAccounts, $ids, &$partialAccounts, $selectedModules, &$type) {
		$errors = [];
		for ($i = 0; $i < count($rawAccounts); $i++) {
			// add object class
			if (!in_array('group', $partialAccounts[$i]['objectClass'])) {
				$partialAccounts[$i]['objectClass'][] = 'group';
			}
			// cn + sAMAccountName
			if ($rawAccounts[$i][$ids['windowsGroup_name']] != "") {
				if (get_preg($rawAccounts[$i][$ids['windowsGroup_name']], 'groupname')) {
					$partialAccounts[$i]['cn'] = $rawAccounts[$i][$ids['windowsGroup_name']];
					$partialAccounts[$i]['sAMAccountName'] = $rawAccounts[$i][$ids['windowsGroup_name']];
				}
				else {
					$errMsg = $this->messages['cn'][1];
					$errMsg[] = [$i];
					$errors[] = $errMsg;
				}
			}
			// description
			$this->mapSimpleUploadField($rawAccounts, $ids, $partialAccounts, $i, 'windowsGroup_description', 'description');
			// notes
			$this->mapSimpleUploadField($rawAccounts, $ids, $partialAccounts, $i, 'windowsGroup_notes', 'info');
			// email
			if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemail')) {
				$this->mapSimpleUploadField($rawAccounts, $ids, $partialAccounts, $i, 'windowsGroup_mail', 'mail',
					'email', $this->messages['mail'][1], $errors);
			}
			// managed by
			if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemanagedBy')) {
				$this->mapSimpleUploadField($rawAccounts, $ids, $partialAccounts, $i, 'windowsGroup_managedBy', 'managedBy');
			}
			// add members
			if ($rawAccounts[$i][$ids['windowsGroup_members']] != "") {
				$partialAccounts[$i]['member'] = explode(";", $rawAccounts[$i][$ids['windowsGroup_members']]);
			}
			// group scope
			if ($rawAccounts[$i][$ids['windowsGroup_scope']] != "") {
				if (in_array($rawAccounts[$i][$ids['windowsGroup_scope']], $this->groupScopes)) {
					switch ($rawAccounts[$i][$ids['windowsGroup_scope']]) {
						case windowsGroup::SCOPE_DOMAIN_LOCAL:
							$partialAccounts[$i]['groupType'] = 4;
							break;
						case windowsGroup::SCOPE_GLOBAL:
							$partialAccounts[$i]['groupType'] = 2;
							break;
						case windowsGroup::SCOPE_UNIVERSAL:
							$partialAccounts[$i]['groupType'] = 8;
							break;
					}
				}
				else {
					$errMsg = $this->messages['groupScope'][0];
					$errMsg[] = [$i];
					$errors[] = $errMsg;
				}
			}
			else {
				$partialAccounts[$i]['groupType'] = 2;
			}
			// group type
			if ($rawAccounts[$i][$ids['windowsGroup_type']] != "") {
				if (in_array($rawAccounts[$i][$ids['windowsGroup_type']], $this->groupTypes)) {
					if ($rawAccounts[$i][$ids['windowsGroup_type']] == windowsGroup::TYPE_SECURITY) {
						$partialAccounts[$i]['groupType'] -= 2147483648;
					}
				}
				else {
					$errMsg = $this->messages['groupType'][0];
					$errMsg[] = [$i];
					$errors[] = $errMsg;
				}
			}
			else {
				$partialAccounts[$i]['groupType'] -= 2147483648;
			}
			// NIS name
			if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30Name', true)) {
				$this->mapSimpleUploadField($rawAccounts, $ids, $partialAccounts, $i, 'windowsGroup_msSFU30Name', 'msSFU30Name',
					'groupname', $this->messages['msSFU30Name'][1], $errors);
			}
			// NIS domain
			if (!$this->isBooleanConfigOptionSet('windowsGroup_hidemsSFU30NisDomain', true)) {
				$this->mapSimpleUploadField($rawAccounts, $ids, $partialAccounts, $i, 'windowsGroup_msSFU30NisDomain', 'msSFU30NisDomain');
			}
		}
		return $errors;
	}

	/**
	 * {@inheritDoc}
	 * @see baseModule::get_pdfEntries()
	 */
	public function get_pdfEntries($pdfKeys, $typeId) {
		$return = [];
		$this->addSimplePDFField($return, 'cn', _('Group name'));
		$this->addSimplePDFField($return, 'description', _('Description'));
		$this->addSimplePDFField($return, 'info', _('Notes'));
		$this->addSimplePDFField($return, 'mail', _('Email address'));
		$this->addSimplePDFField($return, 'msSFU30Name', _('NIS name'));
		$this->addSimplePDFField($return, 'msSFU30NisDomain', _('NIS domain'));
		// group type
		$groupType = windowsGroup::TYPE_SECURITY;
		$groupScope = windowsGroup::SCOPE_GLOBAL;
		if (isset($this->attributes['groupType'][0])) {
			if ((((int) $this->attributes['groupType'][0]) & 4) !== 0) {
				$groupScope = windowsGroup::SCOPE_DOMAIN_LOCAL;
			}
			elseif ((((int) $this->attributes['groupType'][0]) & 8) !== 0) {
				$groupScope = windowsGroup::SCOPE_UNIVERSAL;
			}
			if ((((int) $this->attributes['groupType'][0]) & 0x80000000) === 0) {
				$groupType = windowsGroup::TYPE_DISTRIBUTION;
			}
		}
		$groupTypeLabels = array_flip($this->groupTypes);
		$groupType = $groupTypeLabels[$groupType];
		$groupScopeLabels = array_flip($this->groupScopes);
		$groupScope = $groupScopeLabels[$groupScope];
		$this->addPDFKeyValue($return, 'groupScope', _('Group scope'), $groupScope);
		$this->addPDFKeyValue($return, 'groupType', _('Group type'), $groupType);
		// managed by
		if (isset($this->attributes['managedBy'][0])) {
			$managedBy = getAbstractDN($this->attributes['managedBy'][0]);
			$this->addPDFKeyValue($return, 'managedBy', _('Managed by'), $managedBy);
		}
		// members
		if (!empty($this->attributes['member'])) {
			$memberList = [];
			if (isset($this->attributes['member'][0])) {
				$memberList = $this->attributes['member'];
			}
			usort($memberList, compareDN(...));
			$pdfTable = new PDFTable(_('Members'));
			for ($i = 0; $i < count($memberList); $i++) {
				$pdfRow = new PDFTableRow();
				$pdfRow->cells[] = new PDFTableCell($memberList[$i]);
				$pdfTable->rows[] = $pdfRow;
			}
			$this->addPDFTable($return, 'member', $pdfTable);
		}
		// member of
		if (!empty($this->attributes['memberOf'])) {
			$memberOfList = [];
			if (isset($this->attributes['memberOf'][0])) {
				$memberOfList = $this->attributes['memberOf'];
			}
			usort($memberOfList, compareDN(...));
			$pdfTable = new PDFTable(_('Member of'));
			for ($i = 0; $i < count($memberOfList); $i++) {
				$pdfRow = new PDFTableRow();
				$pdfRow->cells[] = new PDFTableCell($memberOfList[$i]);
				$pdfTable->rows[] = $pdfRow;
			}
			$this->addPDFTable($return, 'memberOf', $pdfTable);
		}
		return $return;
	}

	/**
	 * Finds all existing groups.
	 *
	 * @return array group DNs
	 */
	private function findGroups() {
		if ($this->groupCache != null) {
			return $this->groupCache;
		}
		$return = [];
		$types = ['group'];
		$results = searchLDAPByFilter('(objectClass=group)', ['dn'], $types);
		$count = count($results);
		for ($i = 0; $i < $count; $i++) {
			if (isset($results[$i]['dn'])) {
				$return[] = $results[$i]['dn'];
			}
		}
		$this->groupCache = $return;
		return $return;
	}

	/**
	 * Returns a list of modifications which have to be made to the LDAP account.
	 *
	 * Calling this method requires the existence of an enclosing {@link accountContainer}.<br>
	 * <br>
	 *
	 * <br>This function returns an array with 3 entries:
	 * <br>array( DN1 ('add' => array($attr), 'remove' => array($attr), 'modify' => array($attr)), DN2 .... )
	 * <br>DN is the DN to change. It is possible to change several DNs (e.g. create a new user and add him
	 * to some groups via attribute memberUid)<br>
	 * <br><b>"add"</b> are attributes which have to be added to the LDAP entry
	 * <br><b>"remove"</b> are attributes which have to be removed from the LDAP entry
	 * <br><b>"modify"</b> are attributes which have to be modified in the LDAP entry
	 * <br><b>"notchanged"</b> are attributes which stay unchanged
	 * <br><b>"info"</b> values with informational value (e.g. to be used later by pre/postModify actions)
	 * <br>
	 * <br>This builds the required commands from $this-attributes and $this->orig.
	 *
	 * @return array list of modifications
	 */
	public function save_attributes() {
		$attrs = $this->attributes;
		$orig = $this->orig;
		$attrs['memberOf'] = [];
		$orig['memberOf'] = [];
		return $this->getAccountContainer()->save_module_attributes($attrs, $orig);
	}

	/**
	 * Runs the postmodify actions.
	 *
	 * @param boolean $newAccount
	 * @param array $attributes LDAP attributes of this entry
	 * @return array array which contains status messages. Each entry is an array containing the status message parameters.
	 * @see baseModule::postModifyActions()
	 *
	 */
	public function postModifyActions($newAccount, $attributes) {
		$messages = [];
		// set groups
		$groups = $this->findGroups();
		if (!isset($this->orig['memberOf'])) {
			$this->orig['memberOf'] = [];
		}
		if (!isset($this->attributes['memberOf'])) {
			$this->attributes['memberOf'] = [];
		}
		$toAdd = array_values(array_diff($this->attributes['memberOf'], $this->orig['memberOf']));
		$toRem = array_values(array_diff($this->orig['memberOf'], $this->attributes['memberOf']));
		// add groups
		for ($i = 0; $i < count($toAdd); $i++) {
			if (in_array($toAdd[$i], $groups)) {
				$success = @ldap_mod_add($_SESSION['ldap']->server(), $toAdd[$i], ['member' => [$this->getAccountContainer()->finalDN]]);
				if (!$success) {
					logNewMessage(LOG_ERR, 'Unable to add group ' . $this->getAccountContainer()->finalDN . ' to group: ' . $toAdd[$i] . ' (' . ldap_error($_SESSION['ldap']->server()) . ').');
					$messages[] = ['ERROR', sprintf(_('Was unable to add attributes to DN: %s.'), $toAdd[$i]), getDefaultLDAPErrorString($_SESSION['ldap']->server())];
				}
				else {
					logNewMessage(LOG_NOTICE, 'Added group ' . $this->getAccountContainer()->finalDN . ' to group: ' . $toAdd[$i]);
				}
			}
		}
		// remove groups
		for ($i = 0; $i < count($toRem); $i++) {
			if (in_array($toRem[$i], $groups)) {
				$success = @ldap_mod_del($_SESSION['ldap']->server(), $toRem[$i], ['member' => [$this->getAccountContainer()->dn_orig]]);
				if (!$success) {
					logNewMessage(LOG_ERR, 'Unable to delete group ' . $this->getAccountContainer()->finalDN . ' from group: ' . $toRem[$i] . ' (' . ldap_error($_SESSION['ldap']->server()) . ').');
					$messages[] = ['ERROR', sprintf(_('Was unable to remove attributes from DN: %s.'), $toRem[$i]), getDefaultLDAPErrorString($_SESSION['ldap']->server())];
				}
				else {
					logNewMessage(LOG_NOTICE, 'Removed group ' . $this->getAccountContainer()->finalDN . ' from group: ' . $toRem[$i]);
				}
			}
		}
		return $messages;
	}

	/**
	 * Recursively gets the members of this group and its subgroups.
	 *
	 * @return list of DNs
	 */
	private function getEffectiveMembers() {
		$membersToCheck = $this->attributes['member'];
		$effectiveMembers = $membersToCheck;
		while (!empty($membersToCheck)) {
			$member = array_pop($membersToCheck);
			$attrs = ldapGetDN($member, ['member']);
			if (!empty($attrs['member'])) {
				foreach ($attrs['member'] as $newMember) {
					if (!in_array($newMember, $effectiveMembers)) {
						$effectiveMembers[] = $newMember;
						$membersToCheck[] = $newMember;
					}
				}
			}
		}
		return $effectiveMembers;
	}

	/**
	 * {@inheritDoc}
	 * @see baseModule::get_configOptions()
	 */
	public function get_configOptions($scopes, $allScopes) {
		$configContainer = new htmlResponsiveRow();
		$configContainerHead = new htmlGroup();
		$configContainerHead->addElement(new htmlOutputText(_('Hidden options')));
		$configContainerHead->addElement(new htmlHelpLink('hiddenOptions'));
		$configContainer->add($configContainerHead);
		$configContainer->add(new htmlResponsiveInputCheckbox('windowsGroup_hidemail', false, _('Email address'), null, true), 12, 4);
		$configContainer->add(new htmlResponsiveInputCheckbox('windowsGroup_hidemanagedBy', false, _('Managed by'), null, true), 12, 4);
		$configContainer->add(new htmlResponsiveInputCheckbox('windowsGroup_hidemsSFU30Name', true, _('NIS name'), null, true), 12, 4);
		$configContainer->add(new htmlResponsiveInputCheckbox('windowsGroup_hidemsSFU30NisDomain', true, _('NIS domain'), null, true), 12, 4);
		for ($i = 0; $i < 2; $i++) {
			$configContainer->add(new htmlOutputText(''), 0, 4);
		}
		return $configContainer;
	}

	/**
	 * @inheritDoc
	 */
	public function getListAttributeDescriptions(ConfiguredType $type): array {
		return [
			'cn' => _('Group name'),
			'description' => _('Description'),
			'info' => _('Notes'),
			'member' => _('Members'),
			'memberof' => _('Member of'),
			'managedby' => _('Managed by'),
			'grouptype' => _('Group type'),
			'groupscope' => _('Group scope'),
			'whencreated' => _('Creation time'),
			'whenchanged' => _('Change date'),
		];
	}

	/**
	 * @inheritDoc
	 */
	public function getListRenderFunction(string $attributeName): ?callable {
		if (($attributeName === 'whencreated') || ($attributeName === 'whenchanged')) {
			return function(array $entry, string $attribute): htmlElement {
				$value = ' - ';
				if (!empty($entry[$attribute][0])) {
					$value = formatLDAPTimestamp($entry[$attribute][0]);
				}
				return new htmlOutputText($value);
			};
		}
		elseif (($attributeName === 'managedby') || ($attributeName === 'member')) {
			return function(array $entry, string $attribute): htmlElement {
				if (!empty($entry[$attribute][0])) {
					$typeManager = new TypeManager();
					$values = $entry[$attribute];
					if (!empty($values)) {
						usort($values, compareDN(...));
					}
					$count = count($values);
					for ($i = 0; $i < $count; $i++) {
						$replaced = false;
						foreach ($typeManager->getConfiguredTypes() as $type) {
							if ((stripos($values[$i], $type->getSuffix()) > 0) && !isAccountTypeHidden($type->getId())) {
								$values[$i] = '<a href="../account/edit.php?type=' . $type->getId() . '&amp;DN=\'' . $values[$i] . '\'">' . getAbstractDN($values[$i]) . '</a>';
								$replaced = true;
								break;
							}
						}
						if (!$replaced) {
							$values[$i] = getAbstractDN($values[$i]);
						}
					}
					return new htmlDiv(null, new htmlOutputText(implode('<br>', $values), false), ['rightToLeftText']);
				}
				return new htmlGroup();
			};
		}
		return null;
	}

}
